<?php

namespace ZapperIntegration\Payments\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\Data\TransactionInterface;
use Magento\Sales\Api\InvoiceRepositoryInterface;
use Magento\Sales\Api\OrderPaymentRepositoryInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Api\TransactionRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Payment\Transaction;
use Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface;


class PaymentHandler
{
    private ZapperApiInterface $paymentHistory;
    private InvoiceRepositoryInterface $invoiceRepository;
    private OrderPaymentRepositoryInterface $paymentRepository;
    private OrderRepositoryInterface $orderRepository;
    private ScopeConfigInterface $scopeConfig;
    private TransactionRepositoryInterface $transactionRepository;
    private Transaction\BuilderInterface $transactionBuilder;

    /**
     * @param InvoiceRepositoryInterface $invoiceRepository
     * @param OrderPaymentRepositoryInterface $paymentRepository
     * @param OrderRepositoryInterface $orderRepository
     * @param ScopeConfigInterface $scopeConfig
     * @param TransactionRepositoryInterface $transactionRepository
     * @param BuilderInterface $transactionBuilder
     * @param ZapperApiInterface $paymentHistory
     */
    public function __construct(
        InvoiceRepositoryInterface $invoiceRepository,
        OrderPaymentRepositoryInterface $paymentRepository,
        OrderRepositoryInterface $orderRepository,
        ScopeConfigInterface $scopeConfig,
        TransactionRepositoryInterface $transactionRepository,
        Transaction\BuilderInterface $transactionBuilder,
        ZapperApiInterface $paymentHistory
    ) {
        $this->invoiceRepository = $invoiceRepository;
        $this->paymentRepository = $paymentRepository;
        $this->orderRepository = $orderRepository;
        $this->scopeConfig = $scopeConfig;
        $this->transactionRepository = $transactionRepository;
        $this->transactionBuilder = $transactionBuilder;
        $this->paymentHistory = $paymentHistory;
    }

    public function HandlePayment($zapperId): bool
    {
        $paymentInfo = $this->paymentHistory->fetchPayment($zapperId);

        if ($paymentInfo[ZapperConstants::STATUS] != 1) {
            return false;
        }


        $orderId = $paymentInfo[ZapperConstants::ORDER_ID];
        $order = $this->orderRepository->get($orderId);

        if ($paymentInfo[ZapperConstants::AMOUNT] < $order->getGrandTotal()) {
            return false;
        }

        $this->createPayment($order, $paymentInfo);
        return true;
    }

    private function createPayment(OrderInterface $order, array $paymentInfo): void
    {

        $defaultStatus = Order::STATE_PROCESSING;

        if ($this->scopeConfig->isSetFlag(ZapperConstants::CONFIG_DEFAULT_STATUS)) {
            $defaultStatus = $this->scopeConfig->getValue(ZapperConstants::CONFIG_DEFAULT_STATUS);
        }

        $amount = $paymentInfo[ZapperConstants::AMOUNT];
        $zapperId = $paymentInfo[ZapperConstants::ZAPPER_ID];

        $formattedPrice = $order->getBaseCurrency()->formatTxt(
            $amount
        );

        // payment
        $payment = $this->paymentRepository->create();
        $payment->setMethod("zapper");
        $payment->setAdditionalData("Zapper:$zapperId");
        $payment->setAdditionalInformation('metadata', ['zapperId' => $zapperId]);
        $payment->setAmountPaid($amount);
        $payment->setAmountAuthorized($amount);
        $payment->setLastTransId($zapperId);


        $invoice = $this->invoiceRepository->create();
        $invoice->setOrderId($order->getEntityId());
        $invoice->setGrandTotal($amount / 100);
        $invoice->pay();
        $billingAddressId = $invoice->getBillingAddressId();
        $invoice->setShippingAddressId($billingAddressId);
        $this->invoiceRepository->save($invoice);

        // order status update
        $order->setState($defaultStatus);
        $order->setStatus($defaultStatus);
        // order payment amount update
        $order->setPayment($payment);
        $order->setTotalPaid($amount / 100);


        $this->paymentRepository->save($payment);
        $this->orderRepository->save($order);

        // transaction
        $trans = $this->transactionBuilder;
        $transaction = $trans->setPayment($payment)
            ->setOrder($order)
            ->setTransactionId($zapperId)
            ->setAdditionalInformation(
                [Transaction::RAW_DETAILS => (array) $paymentInfo]
            )
            ->setFailSafe(true)
            //build method creates the transaction and returns the object
            ->build(TransactionInterface::TYPE_CAPTURE);


        $message = "The authorized amount is $formattedPrice.";
        $payment->addTransactionCommentsToOrder(
            $transaction,
            $message
        );
        $this->transactionRepository->save($transaction);
    }

}