<?php

namespace ZapperIntegration\Payments\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\HTTP\Client\Curl;

class ZapperApi implements ZapperApiInterface
{

    private Curl $curl;
    private ScopeConfigInterface $scopeConfig;

    /**
     * @param Curl $curl
     * @param ScopeConfigInterface $scopeConfig
     */
    public function __construct(Curl $curl, ScopeConfigInterface $scopeConfig)
    {
        $this->curl = $curl;
        $this->scopeConfig = $scopeConfig;
    }

    private function getUrl() : string {
        return getenv("ZAPPER_ENDPOINT") ?: ZapperConstants::API_BASE_URL;
    }

    function fetchPayment(string $zapperId): array
    {
        // fetch from zapper api
        $merchantId = $this->scopeConfig->getValue(ZapperConstants::CONFIG_MERCHANT_ID);
        $apiKey = $this->scopeConfig->getValue(ZapperConstants::CONFIG_APIKEY);

        $this->curl->setHeaders(["Authorization" => "Bearer $apiKey"]);
        $baseUrl = $this->getUrl();

        $this->curl->get("$baseUrl/api/v1/merchants/$merchantId/payments/$zapperId");

        $statusCode = $this->curl->getStatus();

        // default response
        $response = array();
        $response[ZapperConstants::STATUS] = false;
        $response[ZapperConstants::AMOUNT] = 0;


        // ensure successful call
        if ($statusCode < 100 || $statusCode > 200) {
            return $response;
        }

        // decode information
        $content = $this->curl->getBody();
        $paymentHistory = json_decode($content);

        if (count($paymentHistory) != 1) {
            return $response;
        }

        // set response
        $payment = get_object_vars($paymentHistory[0]);

        $response[ZapperConstants::ZAPPER_ID] = $zapperId;
        $response[ZapperConstants::STATUS] = $payment["status"] == 1;
        $response[ZapperConstants::AMOUNT] = $payment["tenderedAmount"];
        $response[ZapperConstants::ORDER_ID] = $payment["invoiceExternalReference"];

        return $response;
    }

    function testCredentials(string $merchantId, string $siteId, string $token): bool
    {
        if (empty($merchantId)) {
            return false;
        }

        if (empty($siteId)) {
            return false;
        }

        if (empty($token)) {
            return false;
        }

        $this->curl->setHeaders(["Authorization" => "Bearer $token"]);
        $baseUrl = $this->getUrl();
        $this->curl->get("$baseUrl/api/v1/merchants/$merchantId/sites/$siteId/payments");
        $statusCode = $this->curl->getStatus();
        return $statusCode >= 200 && $statusCode < 300;
    }
}
