<?php

namespace ZapperIntegration\Payments\Controller\Payments;

use Magento\Checkout\Model\Session;
use Magento\Framework\App\Action\HttpGetActionInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Store\Model\StoreManagerInterface;
use ZapperIntegration\Payments\Model\UrlHelper;
use ZapperIntegration\Payments\Model\ZapperConstants;
use Psr\Log\LoggerInterface;

class Checkout implements HttpGetActionInterface
{
    private Curl $curl;
    private ResultFactory $resultFactory;
    private Session $checkoutSession;
    private ScopeConfigInterface $scopeConfig;
    private StoreManagerInterface $storeManager;

    private LoggerInterface $logger;

    /**
     * @param Curl $curl
     * @param ResultFactory $resultFactory
     * @param Session $checkoutSession
     * @param ScopeConfigInterface $scopeConfig
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(Curl $curl, ResultFactory $resultFactory, Session $checkoutSession, ScopeConfigInterface $scopeConfig, StoreManagerInterface $storeManager, LoggerInterface $logger)
    {
        $this->curl = $curl;
        $this->resultFactory = $resultFactory;
        $this->checkoutSession = $checkoutSession;
        $this->scopeConfig = $scopeConfig;
        $this->storeManager = $storeManager;
        $this->logger = $logger;
    }

    /**
     * Execute action based on request and return result
     *
     * @return ResultInterface
     * @throws NoSuchEntityException
     */
    public function execute() : ResultInterface
    {
        $this->logger->debug("Zapper Payments Checkout: entered execute");
        $response = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);

        $order = $this->checkoutSession->getLastRealOrder();

        $merchantId = $this->scopeConfig->getValue(ZapperConstants::CONFIG_MERCHANT_ID);
        $siteId = $this->scopeConfig->getValue(ZapperConstants::CONFIG_SITE_ID);

        $store = $this->storeManager->getStore();
        $baseUrl = $store->getBaseUrl();
        $storeName = $store->getName();

        $body = $this->createBody(
            $order->getBaseGrandTotal(),
            $merchantId,
            $siteId,
            $baseUrl,
            $order->getId(),
            $storeName,
            $order->getBillingAddress() ? $order->getBillingAddress()->getFirstname() : null
        );

        $headers = ["User-Agent" => "magento"];
        $this->curl->setHeaders($headers);
        
        $this->logger->debug("Zapper Payments Checkout: about to call to: " . $this->getUrl() . "api/magento/session");
        $this->curl->setTimeout(ZapperConstants::ECOMMERCE_GATEWAY_TIMEOUT_SECONDS);
        $this->curl->post($this->getUrl() . "api/magento/session", $body);
        $status = $this->curl->getStatus();
        if ($status != 200) {
            $this->logger->debug("Zapper Payments Checkout: status not 200, but is: " . $status);
            $this->logger->debug("Zapper Payments Checkout: response body is: " . $this->curl->getBody());
            $response->setUrl($body["cancelUrl"]);
            return $response;
        }

        $responseBody = json_decode($this->curl->getBody(), true);
        $response->setUrl($responseBody["redirectUrl"]);
        return $response;
    }

    private function getUrl() : string {
        return getenv(ZapperConstants::ENV_VAR_ZAPPER_ECOM_GATEWAY) ?: ZapperConstants::ECOMMERCE_GATEWAY_URL;
    }

    private function createBody(
        $amount,
        $merchant,
        $site,
        $baseUrl,
        $orderId,
        $name,
        $customerName
    ) : array {
        return [
            "amount" => $amount,
            "merchantId" => $merchant,
            "siteId" => $site,
            "notificationUrl" => UrlHelper::buildUrl([$baseUrl, ZapperConstants::API_NOTIFICATION]),
            "cancelUrl" =>  UrlHelper::buildUrl([$baseUrl, ZapperConstants::API_CANCEL]),
            "completeUrl" =>  sprintf("%s?orderId=%s", UrlHelper::buildUrl([$baseUrl, ZapperConstants::API_COMPLETE]), $orderId),
            "reference" => $orderId,
            "siteName" => $name,
            "customerName" => $customerName
        ];
    }
}
